package easik.sketch.util.Export.Constraints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Set;

import easik.sketch.util.Export.ExportConstants;

/**
 * A class to format all triggers for SQL exportation.  There are three different types of triggers which are created: insert, 
 * update, and delete triggers.  
 * 
 * @author Vera Ranieri 2006
 * @since 2006-06-16 Vera Ranieri
 * @version 2006-07-13 Vera Ranieri
 */
public class ExportTriggerFormatter {
	/**
	 * SQL statement to set trigger to activate before insert
	 */
	public final static String BEFORE_INSERT_TIME = "BEFORE INSERT ON ";
	/**
	 * SQL statement to set trigger to activate after insert
	 */
	public final static String AFTER_INSERT_TIME = "AFTER INSERT ON ";
	/**
	 * SQL statement to set trigger to activate before delete
	 */
	public final static String BEFORE_DELETE_TIME = "BEFORE DELETE ON ";
	
	
	/**
	 * Hash map of all before insert procedures, indexed by the table to which they apply
	 */
	private HashMap<String, LinkedList<ExportConstraint>> _beforeInsertProcedures;
	/**
	 * Hash map of all after insert procedures, indexed by the table to which they apply
	 */
	private HashMap<String, LinkedList<ExportConstraint>> _afterInsertProcedures;
	/**
	 * Hash map of all before delete procedures, indexed by the table to which they apply
	 */
	private HashMap<String, LinkedList<ExportConstraint>> _beforeDeleteProcedures;
	/**
	 * List of all trigger strings formatted for SQL
	 */
	private ArrayList<String> _triggerStrings;
	/**
	 * Constructor.  Initialises all fields so new triggers can be added.
	 */
	public ExportTriggerFormatter(){
		_beforeInsertProcedures = new HashMap<String, LinkedList<ExportConstraint>>();
		_triggerStrings = new ArrayList<String>();
		
		_afterInsertProcedures = new HashMap<String, LinkedList<ExportConstraint>>();
		
		_beforeDeleteProcedures = new HashMap<String, LinkedList<ExportConstraint>>();
		
	}
	
	/**
	 * Sets all triggers by determining all procedures that must be called from inside a each respective trigger.
	 */
	public void setTriggers(){
		setTimeTriggers(_beforeInsertProcedures, "BInsert", BEFORE_INSERT_TIME, "(NEW.");
		setTimeTriggers(_afterInsertProcedures, "AInsert", AFTER_INSERT_TIME, "(NEW.");
		setTimeTriggers(_beforeDeleteProcedures, "BDelete", BEFORE_DELETE_TIME, "(OLD.");
	}
	
	/**
	 * Adds information about the requirement for a trigger. This information is added to a Hashmap that is indexed by 
	 * the name of the table on which the trigger must be called.  The type of trigger needed to be added is determined
	 * by the type of constraint being added.
	 * 
	 * @param con The constraint for which a trigger must be enforced.
	 */
	public void addTrigger(ExportConstraint con){
		if(con instanceof ExportCommutativeDiagram){
			
			addTimeTrigger(AFTER_INSERT_TIME, _afterInsertProcedures, con);
		}
		else if(con instanceof ExportSumConstraint){
			addTimeTrigger(BEFORE_INSERT_TIME, _beforeInsertProcedures, con);
			addTimeTrigger(BEFORE_DELETE_TIME, _beforeDeleteProcedures, con);
			
		}
		else if(con instanceof ExportProductConstraint){
			addTimeTrigger(AFTER_INSERT_TIME, _afterInsertProcedures, con);
			addTimeTrigger(BEFORE_DELETE_TIME, _beforeDeleteProcedures, con);
			
		}
		else if(con instanceof ExportPullback){
			addTimeTrigger(AFTER_INSERT_TIME, _afterInsertProcedures, con);
			addTimeTrigger(BEFORE_DELETE_TIME, _beforeDeleteProcedures, con);
		}
	}
	
	/**
	 * Returns an arraylist of all trigger strings.  These strings are formatted for SQL. 
	 * 
	 * @return An arraylist of all trigger strings for the schema.
	 */
	public ArrayList<String> getAllTriggers(){
		return _triggerStrings;
	}
	

	/**
	 * Records constraint to be activated at the given time.  Constraints are stored in a Hashmap indexed by the name 
	 * of the table to which the constraint applies.  This is done so all constraints can be enforced within one 
	 * trigger.
	 * 
	 * @param time The time at which the constraint should be enforced
	 * @param triggers The current hash map of triggers, specific for that time.
	 * @param c The constraint to be added to the hash map of triggers
	 * @since 2006-07-07 Vera Ranieri
	 */
	private void addTimeTrigger(String time, HashMap<String, LinkedList<ExportConstraint>> triggers, ExportConstraint c){
		
		Set<String> i = c.getTimeNames(time).keySet();
		LinkedList<ExportConstraint> constraints;
		for(String table : i){
			constraints = triggers.get(table);
			if(constraints != null){
				constraints.add(c);
			}
			else{
				constraints = new LinkedList<ExportConstraint>();
				constraints.add(c);
				triggers.put(table, constraints);
			}
		}
	}
	
	

	/**
	 * Sets all triggers based on the <it>time</it> passed as a parameter
	 * 
	 * @param triggers The triggers which need to be set
	 * @param name The unique name of the trigger
	 * @param time The time at which the trigger is to be called
	 * @param procVar The variables which must be passed when formatting the <it>call</it> string.
	 * @since 2006-07-05 Vera Ranieri
	 */
	private void setTimeTriggers(HashMap<String, LinkedList<ExportConstraint>> triggers, String name, String time, String procVar){
		Set<String> i = triggers.keySet();
		String trigger;
		LinkedList<ExportConstraint> l;
		HashMap<String, String> trigs;
		boolean exists;
		for(String table: i){
			exists = false;
			
			trigger = "CREATE TRIGGER " + table+ name+"Trig " + time +table + 
				" FOR EACH ROW " + ExportConstants.BEGIN;
			l = triggers.get(table);
			
			for(ExportConstraint c : l ){
				trigs = c.getTimeNames(time);
				
				String procName = trigs.get(table);
				if(procName != null){
					if(c instanceof ExportSumConstraint)
						trigger += "call " + procName + procVar + ((ExportSumConstraint)c).getVarTable(table) + ExportConstants.ID+"); \n";
					
					else
						trigger +="call "+ procName + procVar + table + ExportConstants.ID+"); \n";
						
					
					exists = true;
				}
					
				
			}
			trigger += ExportConstants.END;
			
			if(exists)
				_triggerStrings.add(trigger);
		}
	}
}
